#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <locale.h>

#define VERSION "MEAN Version 0.2"
#define DATE    "31 October 2025"
#define AUTHOR  "Michael J. Chappell"

#if !defined(WIDTH)
#define WIDTH   16
#endif

#if !defined(PREC)
#define PREC    3
#endif

struct list_t {
  double val;
  struct list_t *ptr;
};

void help(void) {
  printf("%s\n", VERSION);
  printf("%s\n", AUTHOR);
  printf("%s\n", DATE);
  printf("\n");
  printf("mean [-d]|[-g]|[-G]|[-h]|[-m]|[-p]|[-q]|[-s]|[-v]\n");
  printf("\n");
  printf("Prints the sum, product, arithmetic and geometric means and\n");
  printf("standard deviation for a list of items.\n");
  printf("\n");
  printf("OPTIONS:\n");
  printf("\n");
  printf("\t-d) Prints the standard deviation.\n");
  printf("\t-g) Prints the geometric mean.\n");
  printf("\t-G) Prints the geometric mean using logarithms.\n");
  printf("\t-h) Prints the help message.\n");
  printf("\t-m) Prints the arithmetic mean (average).\n");
  printf("\t-p) Prints the product.\n");
  printf("\t-q) Surpresses the prompt.\n");
  printf("\t-s) Prints the sum.\n");
  printf("\t-v) Prints the version.\n");
  printf("\n");
  printf("Options suppress prompt.\n");
  printf("\n");
  printf("RETURNS 0 on success, 1 on invalid option, 2 on other errors.\n");
  printf("\n");

  return;
}

void version(void) {
  printf("%s\n", VERSION);
  printf("%s\n", AUTHOR);
  printf("%s\n", DATE);
  printf("Compiled: %s\n", __DATE__);
  printf("\n");

  return;
}

int main(int argc, char **argv) {
  struct list_t *phead, *pnext, *p;
  int i, count;
  char buffer[64];
  double sum, product, logsum;
  double mean, geomean, loggeomean;
  double min, max;
  double stddev, sum_squares;

  int verbose, quiet_opts;
  int pdev, pgeom, pgeomwl, pmean, pprod, plogsum, psum;
  
  phead       = NULL;
  count       = 0;
  sum         = 0.0l;
  logsum      = 0.0l;
  product     = 1.0l;
  sum_squares = 0.0l;

  verbose     = 1;
  pdev        = 0;
  pgeom       = 0;
  pgeomwl     = 0;
  pmean       = 0;
  pprod       = 0;
  plogsum     = 0;
  psum        = 0;

  if (argc>2) {
    fprintf(stderr, "Incorrect number of arguments\n\n");
    help();
    return 1;
  }

  
  if (argc>1) {
    for (i=1; i<argc; i++) {
      switch (argv[i][1]) {
      case 'd':
	pdev = 1;
	break;
      case 'g':
	pgeom = 1;
	break;
      case 'G':
	pgeomwl =1;
	break;
      case 'h':
	help();
	return EXIT_SUCCESS;
	break;
      case 'm':
	pmean = 1;
	break;
      case 'p':
	pprod = 1;
	break;
      case 's':
	psum = 1;
	break;
      case 'v':
	version();
	return EXIT_SUCCESS;
	break;
      case 'q':
	verbose=0;
	break;
      default:
	fprintf(stderr, "Option: \'%s\' not recognized.\n\n", argv[1]);
	help();
	return 1;
      }
    }
  }

  quiet_opts = pdev + pgeom + pgeomwl + pmean + pprod + plogsum + psum;
  if (quiet_opts) verbose = 0;


  setlocale(LC_ALL, "en_US.UTF-8");
  
  if (verbose) {
    printf("Enter values, terminate with Ctrl-D:\n");
  }
  while (fgets(buffer, 63, stdin) != NULL) {
    count++;
    if (pnext = malloc(sizeof(struct list_t))) {
      pnext->ptr = NULL;
      pnext->val = strtod(buffer, NULL);
      if (phead) {
	p->ptr = pnext;
      } else {
	phead = pnext;
      }
    } else {
      fprintf(stderr, "Out of memory!");
      return EXIT_FAILURE;
    }
    p = pnext;
  }

  if (!quiet_opts) printf("ARITHMETIC AND GEOMETRIC MEAN\n");
  if (count==0) {
    fprintf(stderr, "Insufficient number of values.\n");
    return 2;
  }
  if (!quiet_opts) printf("Count :\t%d\n", count);
  if (!quiet_opts) printf("Values:\n");
  min = phead->val;
  max = min;
  while (phead) {
    if (!quiet_opts) printf("\t%'*.*lf\n", WIDTH, PREC, phead->val);
    min = (phead->val)<min?(phead->val):min;
    max = (phead->val)>max?(phead->val):max;
    sum += phead->val;
    product *= phead->val;
    logsum += log(phead->val);
    sum_squares = sum_squares+(phead->val)*(phead->val);
    p = phead;
    phead = p->ptr;
    free(p);
  }

  mean = sum/count;
  geomean = pow(product, 1.0l/count);
  loggeomean = exp(logsum/count);
  stddev = sqrt((sum_squares-sum*sum/count)/(count-1));

  if (!quiet_opts) {
    printf("\n");
    printf("Range             : %'*.*lf: %'0.*lf...%'0.*lf\n",
	   WIDTH, PREC, max-min, PREC, min, PREC, max);
    printf("Sum               : %'*.*lf\n", WIDTH, PREC, sum);
    printf("Average           : %'*.*lf\n", WIDTH, PREC, mean);
    printf("\n");
    printf("Product           : %'*.*lf\n", WIDTH, PREC, product);
    printf("Geometric mean    : %'*.*lf\n", WIDTH, PREC, geomean);
    printf("Standard deviation: %'*.*lf\n", WIDTH, PREC, stddev);
    printf("\n");
    printf("Sum of Logs       : %'*.*lf\n", WIDTH, PREC, logsum);
    printf("Geo. mean w/logs  : %'*.*lf\n", WIDTH, PREC, loggeomean);
  }

  if (pdev) printf("%'*.*lf\n", WIDTH, PREC, stddev);
  if (pprod) printf("%'*.*lf\n", WIDTH, PREC, product);
  if (psum) printf("%'*.*lf\n", WIDTH, PREC, sum);
  if (pmean) printf("%'*.*lf\n", WIDTH, PREC, mean);
  if (pgeom) printf("%'*.*lf\n", WIDTH, PREC, geomean);
  if (pgeomwl) printf("%'*.*lf\n", WIDTH, PREC, loggeomean);

  return EXIT_SUCCESS;
}
